<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Theme Functions
 *
 * General theme functions.
 *
 * @since   1.0.0
 * @author 	WooThemes
 */

/**
 * Filter WooFramework pagination.
 *
 * @since  	1.0.0
 * @return 	array
 * @param   array $args WooFramework pagination args
 * @link 	http://docs.woothemes.com/document/woo_pagination/
 * @author 	WooThemes
 */
function woo_pagination_html5_markup ( $args ) {
	$args['before'] = '<nav class="pagination woo-pagination">';
	$args['after'] 	= '</nav>';

	return $args;
} // End woo_pagination_html5_markup()
add_filter( 'woo_pagination_args', 'woo_pagination_html5_markup', 2 );

/**
 * Check for IE.
 *
 * @since  	1.0.0
 * @return 	bool
 * @param   $version IE version
 * @uses  	substr()
 * @author 	WooThemes
 */
if ( ! function_exists( 'is_ie' ) ) {
	function is_ie ( $version = '6.0' ) {
		$supported_versions = array( '6.0', '7.0', '8.0', '9.0' );
		$agent 				= substr( $_SERVER['HTTP_USER_AGENT'], 25, 4 );
		$current_version 	= substr( $_SERVER['HTTP_USER_AGENT'], 30, 3 );
		$response 			= false;

		if ( in_array( $version, $supported_versions ) && 'MSIE' == $agent && ( $version == $current_version ) ) {
			$response = true;
		}

		return $response;
	} // End is_ie()
}

/**
 * Check if WooCommerce is activated.
 *
 * @since  	1.0.0
 * @return 	bool
 * @uses  	class_exists()
 * @author 	WooThemes
 */
if ( ! function_exists( 'is_woocommerce_activated' ) ) {
	function is_woocommerce_activated() {
		return class_exists( 'woocommerce' ) ? true : false;
	}
}

/**
 * Retrieves a menu name.
 *
 * Return the menu title from a location.
 *
 * @since  	1.0.0
 * @return 	string
 * @uses  	get_nav_menu_locations(), wp_get_nav_menu_object()
 * @param   $location Menu location
 * @author 	WooThemes
 */
function woo_get_menu_name( $location ) {
    if ( ! has_nav_menu( $location ) ) return false;

    $menus 			= get_nav_menu_locations();
    $menu_title 	= wp_get_nav_menu_object( $menus[$location] )->name;

    return $menu_title;
}

/**
 * Theme Setup.
 *
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * @uses add_theme_support() To add support for automatic feed links.
 * @uses add_editor_style() To style the visual editor.
 */
if ( ! function_exists( 'woothemes_setup' ) ) {
	function woothemes_setup () {
		add_image_size( 'footer', 720, 360, 1 );

		if ( locate_template( 'editor-style.css' ) != '' ) {
			add_editor_style();
		}

		add_theme_support( 'automatic-feed-links' );

		if ( function_exists( 'wp_nav_menu') ) {
			add_theme_support( 'nav-menus' );
			register_nav_menus(
				array(
					'primary-menu' 	=> __( 'Primary Menu', 'woothemes' )
					)
				);
			register_nav_menus(
				array(
					'top-menu' 		=> __( 'Top Menu', 'woothemes' )
					)
				);
		}
	} // End woothemes_setup()
}

/**
 * Enqueue CSS.
 *
 * Enqueue the theme stylesheet and output it in the <head>
 *
 * @since  	1.0.0
 * @return 	void
 * @uses  	wp_register_style(), wp_enqueue_style
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_load_frontend_css' ) ) {
	function woo_load_frontend_css () {
		wp_register_style( 'theme-stylesheet', get_stylesheet_uri() );
		wp_enqueue_style( 'theme-stylesheet' );
		wp_enqueue_style( 'pt-sans', '//fonts.googleapis.com/css?family=PT+Sans:700, 400' );
		wp_enqueue_style( 'asap', '//fonts.googleapis.com/css?family=Asap:400,700,400italic' );


		do_action( 'woothemes_add_css' );
	} // End woo_load_frontend_css()
}

/**
 * Load Meta Tags.
 *
 * Sets the device width and tells IE to use Chrome Frame if it's installed.
 *
 * @since  	1.0.0
 * @return 	string
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_load_responsive_meta_tags' ) ) {
	function woo_load_responsive_meta_tags () {
		$html = '';

		$html .= "\n" . '<!-- Always force latest IE rendering engine (even in intranet) & Chrome Frame -->' . "\n";
		$html .= '<meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1" />' . "\n";

		/* Remove this if not responsive design */
		$html .= "\n" . '<!--  Mobile viewport scale | Disable user zooming as the layout is optimised -->' . "\n";
		$html .= '<meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1">' . "\n";

		echo $html;
	} // End woo_load_responsive_meta_tags()
}

/**
 * Custom Styling.
 *
 * Output custom CSS based on Theme Options.
 *
 * @since  	1.0.0
 * @return 	string $output concatenated CSS
 * @uses  	woo_get_dynamic_values()
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_custom_styling' ) ) {
	function woo_custom_styling() {

		$output = '';
		$settings = array(
						'body_color' 		=> '',
						'body_img' 			=> '',
						'body_repeat' 		=> '',
						'body_pos' 			=> '',
						'body_attachment' 	=> '',
						'link_color' 		=> '',
						'link_hover_color' 	=> '',
						'button_color' 		=> ''
						);
		$settings = woo_get_dynamic_values( $settings );

		if ( is_array( $settings ) ) {

			// Add CSS to output
			if ( $settings['body_color'] != '' ) {
				$output .= '#inner-wrapper, #sidebar .widget h3 span { background: ' . $settings['body_color'] . ' !important; }' . "\n";
			}

			if ( $settings['body_img'] != '' ) {
				$body_image = $settings['body_img'];
				if ( is_ssl() ) { $body_image = str_replace( 'http://', 'https://', $body_image ); }
				$output .= '#inner-wrapper { background-image: url( ' . esc_url( $body_image ) . ' ) !important; }' . "\n";
			}

			if ( ( $settings['body_img'] != '' ) && ( $settings['body_repeat'] != '' ) && ( $settings['body_pos'] != '' ) ) {
				$output .= '#inner-wrapper { background-repeat: ' . $settings['body_repeat'] . ' !important; }' . "\n";
			}

			if ( ( $settings['body_img'] != '' ) && ( $settings['body_pos'] != '' ) ) {
				$output .= '#inner-wrapper { background-position: ' . $settings['body_pos'] . ' !important; }' . "\n";
			}

			if ( ( $settings['body_img'] != '' ) && ( $settings['body_attachment'] != '' ) ) {
				$output .= '#inner-wrapper { background-attachment: ' . $settings['body_attachment'] . ' !important; }' . "\n";
			}

			if ( $settings['link_color'] != '' ) {
				$output .= 'a { color: ' . $settings['link_color'] . ' !important; }' . "\n";
			}

			if ( $settings['link_hover_color'] != '' ) {
				$output .= 'a:hover, .post-more a:hover, .post-meta a:hover, .post p.tags a:hover { color: ' . $settings['link_hover_color'] . ' !important; }' . "\n";
			}

			if ( $settings['button_color'] != '' ) {
				$output .= 'a.button, a.comment-reply-link, #commentform #submit, #contact-page .submit { background: ' . $settings['button_color'] . ' !important; border-color: ' . $settings['button_color'] . ' !important; }' . "\n";
				$output .= 'a.button:hover, a.button.hover, a.button.active, a.comment-reply-link:hover, #commentform #submit:hover, #contact-page .submit:hover { background: ' . $settings['button_color'] . ' !important; opacity: 0.9; }' . "\n";
			}

		} // End If Statement

		// Output styles
		if ( isset( $output ) && $output != '' ) {
			$output = strip_tags( $output );
			$output = "\n" . "<!-- Woo Custom Styling -->\n<style type=\"text/css\">\n" . $output . "</style>\n";
			echo $output;
		}
	} // End woo_custom_styling()
}

/**
 * Custom Typography Styles.
 *
 * Output custom typography selectors based on Theme Options.
 *
 * @since  	1.0.0
 * @return 	void
 * @uses  	global $woo_options array of Theme Options
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_custom_typography' ) ) {
	function woo_custom_typography() {
		// Get options
		global $woo_options;

		// Reset
		$output 				= '';
		$default_google_font 	= false;

		// Type Check for Array
		if ( is_array( $woo_options ) ) {

			// Add Text title and tagline if text title option is enabled
			if ( isset( $woo_options['woo_texttitle'] ) && $woo_options['woo_texttitle'] == 'true' ) {

				if ( $woo_options['woo_font_site_title'] )
					$output .= 'body #wrapper #header .site-title a {'.woo_generate_font_css($woo_options['woo_font_site_title']).'}' . "\n";
				if ( $woo_options['woo_tagline'] == "true" AND $woo_options['woo_font_tagline'] )
					$output .= 'body #wrapper #header .site-description {'.woo_generate_font_css($woo_options[ 'woo_font_tagline']).'}' . "\n";
			}

			if ( isset( $woo_options['woo_typography'] ) && $woo_options['woo_typography'] == 'true' ) {

				if ( isset( $woo_options['woo_font_body'] ) && $woo_options['woo_font_body'] )
					$output .= 'body { '.woo_generate_font_css($woo_options['woo_font_body'], '1.5').' }' . "\n";

				if ( isset( $woo_options['woo_font_nav'] ) && $woo_options['woo_font_nav'] )
					$output .= 'body #wrapper #navigation .nav a { '.woo_generate_font_css($woo_options['woo_font_nav'], '1.4').' }' . "\n";

				if ( isset( $woo_options['woo_font_page_title'] ) && $woo_options['woo_font_page_title'] )
					$output .= 'body #wrapper .page header h1 { '.woo_generate_font_css($woo_options[ 'woo_font_page_title' ]).' }' . "\n";

				if ( isset( $woo_options['woo_font_post_title'] ) && $woo_options['woo_font_post_title'] )
					$output .= 'body #wrapper .post header h1, body #wrapper .post header h1 a:link, body #wrapper .post header h1 a:visited { '.woo_generate_font_css($woo_options[ 'woo_font_post_title' ]).' }' . "\n";

				if ( isset( $woo_options['woo_font_post_meta'] ) && $woo_options['woo_font_post_meta'] )
					$output .= 'body #wrapper .post-meta { '.woo_generate_font_css($woo_options[ 'woo_font_post_meta' ]).' }' . "\n";

				if ( isset( $woo_options['woo_font_post_entry'] ) && $woo_options['woo_font_post_entry'] )
					$output .= 'body #wrapper .entry, body #wrapper .entry p { '.woo_generate_font_css($woo_options[ 'woo_font_post_entry' ], '1.5').' } h1, h2, h3, h4, h5, h6 { font-family: '.stripslashes($woo_options[ 'woo_font_post_entry' ]['face']).', arial, sans-serif; }'  . "\n";

				if ( isset( $woo_options['woo_font_widget_titles'] ) && $woo_options['woo_font_widget_titles'] )
					$output .= 'body #wrapper .widget h3 { '.woo_generate_font_css($woo_options[ 'woo_font_widget_titles' ]).' }'  . "\n";

				if ( isset( $woo_options['woo_font_widget_titles'] ) && $woo_options['woo_font_widget_titles'] )
					$output .= 'body #wrapper .widget h3 { '.woo_generate_font_css($woo_options[ 'woo_font_widget_titles' ]).' }'  . "\n";

				// Component titles
				if ( isset( $woo_options['woo_font_component_titles'] ) && $woo_options['woo_font_component_titles'] )
					$output .= 'body #wrapper .component h2.component-title { '.woo_generate_font_css($woo_options[ 'woo_font_component_titles' ]).' }'  . "\n";

			// Add default typography Google Font
			} else {

				// Load default Google Fonts
				global $default_google_fonts;
				if ( is_array( $default_google_fonts) and count( $default_google_fonts ) > 0 ) :

					$count = 0;
					foreach ( $default_google_fonts as $font ) {
						$count++;
						$woo_options[ 'woo_default_google_font_'.$count ] = array( 'face' => $font );
					}
					$default_google_font = true;

				endif;

			}

		} // End If Statement

		// Output styles
		if (isset($output) && $output != '') {

			// Load Google Fonts stylesheet in HEAD
			if (function_exists( 'woo_google_webfonts')) woo_google_webfonts();

			$output = "\n" . "<!-- Woo Custom Typography -->\n<style type=\"text/css\">\n" . $output . "</style>\n";
			echo $output;

		// Check if default google font is set and load Google Fonts stylesheet in HEAD
		} elseif ( $default_google_font ) {

			// Enable Google Fonts stylesheet in HEAD
			if ( function_exists( 'woo_google_webfonts' ) ) woo_google_webfonts();

		}

	} // End woo_custom_typography()
}

/**
 * Generate Google Font CSS.
 *
 * Check if Google fonts are specified and output the CSS.
 *
 * @since  	1.0.0
 * @return 	void
 * @uses  	array $google_fonts
 * @param   $option Theme Option
 * @param   $em Font size
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_generate_font_css' ) ) {
	function woo_generate_font_css( $option, $em = '1' ) {

		// Test if font-face is a Google font
		global $google_fonts;

		// Type Check for Array
		if ( is_array( $google_fonts ) ) {

			foreach ( $google_fonts as $google_font ) {

				// Add single quotation marks to font name and default arial sans-serif ending
				if ( $option[ 'face' ] == $google_font[ 'name' ] )
					$option[ 'face' ] = "'" . $option[ 'face' ] . "', arial, sans-serif";

			} // END foreach

		} // End If Statement

		if ( !@$option["style"] && !@$option["size"] && !@$option["unit"] && !@$option["color"] )
			return 'font-family: '.stripslashes($option["face"]).';';
		else
			return 'font:'.$option["style"].' '.$option["size"].$option["unit"].'/'.$em.'em '.stripslashes($option["face"]).';color:'.$option["color"].'!important;';
	}
}

/**
 * Filter body_class.
 *
 * Add layout classes to the body tag via the body_class filter.
 *
 * @since  	1.0.0
 * @return 	array $classes layout classes
 * @uses  	global $woo_options array of Theme Options
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_layout_body_class' ) ) {
	function woo_layout_body_class( $classes ) {

		global $woo_options;

		$layout = 'two-col-left';

		if ( isset( $woo_options['woo_site_layout'] ) && ( $woo_options['woo_site_layout'] != '' ) ) {
			$layout = $woo_options['woo_site_layout'];
		}

		// Set main layout on post or page
		if ( is_singular() ) {
			global $post;
			$single = get_post_meta($post->ID, '_layout', true);
			if ( $single != "" AND $single != "layout-default" )
				$layout = $single;
		}

		// Add layout to $woo_options array for use in theme
		$woo_options['woo_layout'] = $layout;

		// Add classes to body_class() output
		$classes[] = $layout;
		return $classes;

	} // End woo_layout_body_class()
}

/**
 * Customise the search form.
 *
 * @since  	1.0.0
 * @return 	var $html modified searchform markup
 * @param  	$html modified searchform markup
 * @author 	WooThemes
 */
if ( ! function_exists( 'woo_customise_search_form' ) ) {
	function woo_customise_search_form ( $html ) {
	  // Add the "search_main" and "fix" classes to the wrapping DIV tag.
	  $html = str_replace( '<form', '<div class="search_main fix"><form', $html );
	  // Add the "searchform" class to the form.
	  $html = str_replace( ' method=', ' class="searchform" method=', $html );
	  // Add the placeholder attribute and CSS classes to the input field.
	  $html = str_replace( ' name="s"', ' name="s" class="field s" placeholder="' . esc_attr( __( 'Search...', 'woothemes' ) ) . '"', $html );
	  // Wrap the end of the form in a closing DIV tag.
	  $html = str_replace( '</form>', '</form></div>', $html );
	  // Add the "search-submit" class to the button.
	  $html = str_replace( ' id="searchsubmit"', ' class="search-submit" id="searchsubmit"', $html );

	  return $html;
	} // End woo_customise_search_form()
}

// Shorten Excerpt text for use in theme
function woo_excerpt($text, $chars = 120) {
	$text = $text." ";
	$text = substr($text,0,$chars);
	$text = substr($text,0,strrpos($text,' '));
	$text = $text."...";
	return $text;
}